<?php
session_start();

require_once 'includes/functions.php';

// Normalizar la URL solicitada
$request = $_SERVER['REQUEST_URI'];
$request = strtok($request, '?'); // Eliminar query params

// Detectar carpeta base (por si no está en la raíz del servidor)
$scriptDir = str_replace('\\', '/', dirname($_SERVER['SCRIPT_NAME']));
if ($scriptDir !== '/') {
    $request = str_replace($scriptDir, '', $request);
}

$request = trim($request, '/');
$parts = explode('/', $request);
$route = $parts[0] ?? '';

// Enrutador Simple (Front Controller)
switch ($route) {
    case '':
    case 'index.php':
    case 'home':
        require 'views/home.php';
        break;

    case 'login':
        if (isset($_SESSION['user'])) {
            header('Location: dashboard');
            exit;
        }

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $user = $_POST['username'] ?? '';
            $pass = $_POST['password'] ?? ''; // En un sistema real, usar password_verify
            if (verify_login($user, $pass)) {
                $_SESSION['user'] = $user;
                header('Location: dashboard');
                exit;
            } else {
                $error = "Credenciales incorrectas (Prueba: admin / 123)";
                require 'views/login.php';
            }
        } else {
            require 'views/login.php';
        }
        break;

    case 'logout':
        session_destroy();
        header('Location: login');
        exit;

    case 'dashboard':
        if (!isset($_SESSION['user'])) {
            header('Location: login');
            exit;
        }

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            // Eliminar campos que no queremos guardar como submit buttons si los hubiera
            $data = $_POST;
            add_new_record($data);
            // Patrón POST-REDIRECT-GET
            header('Location: dashboard');
            exit;
        }

        $records = get_all_records();
        require 'views/dashboard.php';
        break;

    case 'pdf':
        if (!isset($_SESSION['user'])) {
            header('Location: ../login');
            exit;
        }

        $id = $parts[1] ?? 0;
        $record = get_record_by_id($id);

        if (!$record) {
            die("Error: Registro no encontrado.");
        }

        if (!file_exists('lib/fpdf.php')) {
            die("Error: La libreria FPDF no se encuentra en 'lib/fpdf.php'.");
        }

        require('lib/fpdf.php');

        // ── Helper: encode UTF-8 → ISO-8859-1 para FPDF ──────────────────
        function enc($str)
        {
            $str = (string) $str;
            if (function_exists('iconv')) {
                return iconv('UTF-8', 'ISO-8859-1//TRANSLIT', $str);
            }
            return @utf8_decode($str);
        }

        // ── Helper: valor formateado ──────────────────────────────────────
        function fmt($v)
        {
            return ($v === '' || $v === null) ? '-' : (string) $v;
        }

        // ════════════════════════════════════════════════════════════════
        //  CLASE PDF PERSONALIZADA
        // ════════════════════════════════════════════════════════════════
        class CafePDF extends FPDF
        {

            // Paleta de colores
            const C_CAFE = [59, 42, 30];   // #3B2A1E
            const C_VERDE = [62, 107, 72];   // #3E6B48
            const C_OLIVA = [92, 107, 60];   // #5C6B3C
            const C_CREMA = [239, 230, 221];   // #EFE6DD
            const C_CREMA2 = [249, 245, 241];   // fila alternada más clara
            const C_WHITE = [255, 255, 255];
            const C_GRAY = [120, 100, 85];  // texto secundario

            public $recordId = 0;

            // ── ENCABEZADO ────────────────────────────────────────────────
            function Header()
            {
                $imgPath = __DIR__ . '/images/cafe.png';

                // Banda superior café oscuro
                $this->SetFillColor(...self::C_CAFE);
                $this->Rect(0, 0, 210, 38, 'F');

                // Imagen de granos de café (recortada a la banda)
                if (file_exists($imgPath)) {
                    $this->Image($imgPath, 0, 0, 210, 38);
                }

                // Overlay semitransparente sobre la imagen para legibilidad
                $this->SetFillColor(59, 42, 30);
                // (FPDF no soporta alpha nativo; usamos un rect con color sólido a 60% de opacidad simulado)
                // En su lugar ponemos el texto con sombra desplazada

                // Título principal
                $this->SetFont('Arial', 'B', 18);
                $this->SetTextColor(...self::C_CREMA);
                $this->SetXY(0, 8);
                $this->Cell(210, 10, enc('Analizador de Suelos para Cafe'), 0, 1, 'C');

                // Subtítulo
                $this->SetFont('Arial', 'I', 9);
                $this->SetTextColor(239, 220, 200);
                $this->SetX(0);
                $this->Cell(210, 6, enc('Reporte Tecnico de Diagnostico Agronomico'), 0, 1, 'C');

                // Banda verde debajo del header
                $this->SetFillColor(...self::C_VERDE);
                $this->Rect(0, 38, 210, 6, 'F');

                // Metadatos (ID y fecha) en la banda verde
                $this->SetFont('Arial', '', 7);
                $this->SetTextColor(...self::C_WHITE);
                $this->SetXY(10, 39);
                $this->Cell(90, 4, enc('ID de Registro: #' . $this->recordId), 0, 0, 'L');
                $this->SetX(110);
                $this->Cell(90, 4, enc('Generado: ' . date('d/m/Y H:i')), 0, 1, 'R');

                $this->Ln(6);
                $this->SetTextColor(...self::C_CAFE);
            }

            // ── PIE DE PÁGINA ─────────────────────────────────────────────
            function Footer()
            {
                $this->SetY(-16);

                // Línea separadora verde
                $this->SetDrawColor(...self::C_VERDE);
                $this->SetLineWidth(0.5);
                $this->Line(10, $this->GetY(), 200, $this->GetY());
                $this->Ln(2);

                $this->SetFont('Arial', 'I', 7);
                $this->SetTextColor(...self::C_GRAY);
                $this->Cell(95, 5, enc('Analizador de Suelos para Cafe - Reporte Confidencial'), 0, 0, 'L');
                $this->Cell(95, 5, enc('Pagina ' . $this->PageNo() . ' de {nb}'), 0, 0, 'R');
            }

            // ── TÍTULO DE SECCIÓN ─────────────────────────────────────────
            function SectionTitle($title, $color = null)
            {
                if ($color === null)
                    $color = self::C_VERDE;

                $this->SetFillColor(...$color);
                $this->SetTextColor(...self::C_WHITE);
                $this->SetFont('Arial', 'B', 9);
                $this->SetX(10);
                $this->Cell(190, 7, enc('  ' . $title), 0, 1, 'L', true);
                $this->SetTextColor(...self::C_CAFE);
                $this->Ln(2);
            }

            // ── FILA DE DOS CAMPOS ────────────────────────────────────────
            function FieldRow($label1, $val1, $label2 = '', $val2 = '', $shade = false)
            {
                $fillColor = $shade ? self::C_CREMA : self::C_WHITE;
                $colW = 42;  // ancho etiqueta
                $valW = 53;  // ancho valor
                $gap = 5;   // separación entre columnas
                $h = 6.5;
                $x0 = 10;

                // Columna 1
                $this->SetFillColor(...$fillColor);
                $this->SetFont('Arial', 'B', 8);
                $this->SetTextColor(...self::C_CAFE);
                $this->SetX($x0);
                $this->Cell($colW, $h, enc($label1), 0, 0, 'L', true);

                $this->SetFont('Arial', '', 8);
                $this->SetTextColor(80, 60, 45);
                $this->Cell($valW, $h, enc(fmt($val1)), 0, 0, 'L', true);

                // Columna 2 (si existe)
                if ($label2 !== '') {
                    $this->SetX($x0 + $colW + $valW + $gap);
                    $this->SetFont('Arial', 'B', 8);
                    $this->SetTextColor(...self::C_CAFE);
                    $this->Cell($colW, $h, enc($label2), 0, 0, 'L', true);

                    $this->SetFont('Arial', '', 8);
                    $this->SetTextColor(80, 60, 45);
                    $this->Cell($valW, $h, enc(fmt($val2)), 0, 1, 'L', true);
                } else {
                    $this->Ln($h);
                }

                // Línea divisoria muy sutil
                $this->SetDrawColor(220, 210, 200);
                $this->SetLineWidth(0.1);
                $this->Line(10, $this->GetY(), 200, $this->GetY());
            }

            // ── CAMPO ANCHO (una sola columna) ────────────────────────────
            function FieldWide($label, $val, $shade = false)
            {
                $fillColor = $shade ? self::C_CREMA : self::C_WHITE;
                $h = 6.5;

                $this->SetFillColor(...$fillColor);
                $this->SetFont('Arial', 'B', 8);
                $this->SetTextColor(...self::C_CAFE);
                $this->SetX(10);
                $this->Cell(55, $h, enc($label), 0, 0, 'L', true);

                $this->SetFont('Arial', '', 8);
                $this->SetTextColor(80, 60, 45);
                $this->Cell(135, $h, enc(fmt($val)), 0, 1, 'L', true);

                $this->SetDrawColor(220, 210, 200);
                $this->SetLineWidth(0.1);
                $this->Line(10, $this->GetY(), 200, $this->GetY());
            }

            // ── ESPACIO ENTRE SECCIONES ───────────────────────────────────
            function SectionGap()
            {
                $this->Ln(5);
            }
        }

        // ════════════════════════════════════════════════════════════════
        //  GENERAR EL PDF
        // ════════════════════════════════════════════════════════════════
        $pdf = new CafePDF();
        $pdf->AliasNbPages();
        $pdf->recordId = $record['id'];
        $pdf->AddPage();
        $pdf->SetMargins(10, 10, 10);
        $pdf->SetAutoPageBreak(true, 20);

        // ── SECCIÓN 1: Información del Lote ──────────────────────────────
        $pdf->SectionTitle('1. Informacion del Lote y Muestreo');
        $pdf->FieldRow(
            'Area Lote (Ha)',
            $record['area_lote_ha'] ?? '',
            'Profundidad (cm)',
            $record['profundidad_cm'] ?? '',
            false
        );
        $pdf->FieldRow(
            'Densidad (g/cm3)',
            $record['densidad_gcm3'] ?? '',
            '% Sombra',
            $record['sombra_pct'] ?? '',
            true
        );
        $pdf->FieldRow(
            'Dist. entre plantas (m)',
            $record['dist_plantas_m'] ?? '',
            'Dist. entre surcos (m)',
            $record['dist_surcos_m'] ?? '',
            false
        );
        $pdf->FieldRow(
            'N Total Arboles',
            $record['num_arboles'] ?? '',
            '',
            '',
            true
        );
        $pdf->SectionGap();

        // ── SECCIÓN 2: Información del Cultivo ───────────────────────────
        $pdf->SectionTitle('2. Informacion del Cultivo');
        $pdf->FieldRow(
            'Etapa del Cultivo',
            $record['etapa_cultivo'] ?? '',
            'Edad (anos)',
            $record['edad_anos'] ?? '',
            false
        );
        $pdf->FieldRow(
            'Frec. Fertilizacion',
            $record['frecuencia_fertilizacion'] ?? '',
            'Produccion Prom.',
            $record['produccion_promedio'] ?? '',
            true
        );
        $pdf->SectionGap();

        // ── SECCIÓN 3: Propiedades del Suelo ─────────────────────────────
        $pdf->SectionTitle('3. Propiedades del Suelo');

        // Sub-bloque: Físicas y Químicas
        $pdf->SetFont('Arial', 'BI', 7.5);
        $pdf->SetTextColor(92, 107, 60);
        $pdf->SetX(10);
        $pdf->Cell(190, 5, enc('  Propiedades Fisicas y Quimicas'), 0, 1, 'L');
        $pdf->SetTextColor(59, 42, 30);

        $pdf->FieldRow(
            'pH (1:1)',
            $record['ph'] ?? '',
            'C.E. (dS/m)',
            $record['ce'] ?? '',
            false
        );
        $pdf->FieldRow(
            'M. Organica (%)',
            $record['mo'] ?? '',
            'Nitrogeno Total (%)',
            $record['nitro_total'] ?? '',
            true
        );
        $pdf->Ln(2);

        // Sub-bloque: CIC y Bases
        $pdf->SetFont('Arial', 'BI', 7.5);
        $pdf->SetTextColor(92, 107, 60);
        $pdf->SetX(10);
        $pdf->Cell(190, 5, enc('  Capacidad de Intercambio Cationico y Bases (cmol(+)/kg)'), 0, 1, 'L');
        $pdf->SetTextColor(59, 42, 30);

        $pdf->FieldRow(
            'CIC',
            $record['cic'] ?? '',
            'Aluminio (Al)',
            $record['al'] ?? '',
            false
        );
        $pdf->FieldRow(
            'Calcio (Ca)',
            $record['ca'] ?? '',
            'Magnesio (Mg)',
            $record['mg'] ?? '',
            true
        );
        $pdf->FieldRow(
            'Potasio (K)',
            $record['k'] ?? '',
            'Sodio (Na)',
            $record['na'] ?? '',
            false
        );
        $pdf->SectionGap();

        // ── SECCIÓN 4: Macro y Micro Nutrientes ──────────────────────────
        $pdf->SectionTitle('4. Macro y Micro Nutrientes (ppm o mg/kg)', CafePDF::C_OLIVA);
        $pdf->FieldRow(
            'Fosforo (P)',
            $record['p'] ?? '',
            'Azufre (S)',
            $record['s'] ?? '',
            false
        );
        $pdf->FieldRow(
            'Zinc (Zn)',
            $record['zn'] ?? '',
            'Cobre (Cu)',
            $record['cu'] ?? '',
            true
        );
        $pdf->FieldRow(
            'Boro (B)',
            $record['b'] ?? '',
            'Hierro (Fe)',
            $record['fe'] ?? '',
            false
        );
        $pdf->FieldRow(
            'Manganeso (Mn)',
            $record['mn'] ?? '',
            '',
            '',
            true
        );
        $pdf->SectionGap();

        // ── BLOQUE DE METADATOS (fecha, ID) ──────────────────────────────
        $pdf->Ln(3);
        $pdf->SetFillColor(...CafePDF::C_CREMA);
        $pdf->SetDrawColor(...CafePDF::C_VERDE);
        $pdf->SetLineWidth(0.4);
        $pdf->SetX(10);
        $pdf->SetFont('Arial', '', 7.5);
        $pdf->SetTextColor(...CafePDF::C_GRAY);
        $pdf->Cell(
            190,
            6,
            enc('Registro #' . $record['id'] . '   |   Fecha de registro: ' . ($record['created_at'] ?? 'N/A') . '   |   Usuario: ' . ($_SESSION['user'] ?? 'N/A')),
            'T',
            1,
            'C',
            true
        );

        // ── NOTA AL PIE DEL CONTENIDO ─────────────────────────────────────
        $pdf->Ln(2);
        $pdf->SetFont('Arial', 'I', 7);
        $pdf->SetTextColor(160, 140, 120);
        $pdf->SetX(10);
        $pdf->MultiCell(
            190,
            4,
            enc('Este reporte ha sido generado automaticamente por el sistema Analizador de Suelos para Cafe. Los valores presentados corresponden a los datos ingresados por el usuario y deben ser interpretados por un profesional agronomico calificado.'),
            0,
            'C'
        );

        if (ob_get_length())
            ob_clean();
        $pdf->Output('I', 'diagnostico_' . $record['id'] . '.pdf');
        break;

    default:
        http_response_code(404);
        echo "404 - Página no encontrada. Ruta buscada: " . htmlspecialchars($route);
        break;
}
